<?php

declare(strict_types=1);

namespace App\Command\Portal\User;

use App\Entity\User\User;
use App\Event\User\UserPasswordChangedEvent;
use App\Repository\User\UserRepository;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\Messenger\Attribute\AsMessageHandler;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;

#[AsMessageHandler(handles: ChangeUserPassword::class, bus: 'command.bus')]
final readonly class ChangeUserPasswordHandler
{
    public function __construct(
        private UserRepository              $userRepository,
        private UserPasswordHasherInterface $userPasswordHasher,
        private EventDispatcherInterface    $eventDispatcher
    )
    {
    }

    public function __invoke(ChangeUserPassword $changeUserPassword): void
    {
        $user = $this->userRepository->findByEmail($changeUserPassword->getEmail());

        if (!$user instanceof User) {
            throw new NotFoundHttpException();
        }

        $user->setPassword(
            $this->userPasswordHasher->hashPassword($user, $changeUserPassword->getPassword())
        );

        $this->userRepository->add($user, true);

        $this->eventDispatcher->dispatch(new UserPasswordChangedEvent($user, $changeUserPassword->shouldSendEmail()));
    }
}
