<?php

declare(strict_types=1);

namespace App\Command\Portal\User;

use App\Entity\User\User;
use App\Event\User\UserPasswordResetEvent;
use App\Repository\User\UserRepository;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpKernel\Exception\BadRequestHttpException;
use Symfony\Component\Messenger\Attribute\AsMessageHandler;
use SymfonyCasts\Bundle\ResetPassword\Exception\TooManyPasswordRequestsException;
use SymfonyCasts\Bundle\ResetPassword\ResetPasswordHelperInterface;

#[AsMessageHandler(handles: RequestPasswordChange::class, bus: 'command.bus')]
final readonly class RequestPasswordChangeHandler
{
    public function __construct(
        private UserRepository               $userRepository,
        private ResetPasswordHelperInterface $resetPasswordHelper,
        private EventDispatcherInterface     $eventDispatcher
    )
    {
    }

    public function __invoke(RequestPasswordChange $requestPasswordChange): void
    {
        $user = $this->userRepository->findByEmail($requestPasswordChange->getEmail());

        if (!$user instanceof User) {
            return;
        }

        try {
            $resetPasswordToken = $this->resetPasswordHelper->generateResetToken($user);
        } catch (TooManyPasswordRequestsException $tooManyPasswordRequestsException) {
            throw new BadRequestHttpException($tooManyPasswordRequestsException->getReason());
        }

        $this->eventDispatcher->dispatch(new UserPasswordResetEvent($user, $resetPasswordToken));
    }
}
