<?php

declare(strict_types=1);

namespace App\Command\Portal\User;

use App\Exception\View\ManualConstraintViolationException;
use App\Repository\User\UserRepository;
use App\Service\Url\PortalUrlGeneratorInterface;
use Symfony\Bridge\Twig\Mime\TemplatedEmail;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\Messenger\Attribute\AsMessageHandler;
use Symfony\Contracts\Translation\TranslatorInterface;

#[AsMessageHandler(handles: ResendVerification::class, bus: 'command.bus')]
final readonly class ResendVerificationHandler
{
    public function __construct(
        private UserRepository              $userRepository,
        private PortalUrlGeneratorInterface $portalUrlGenerator,
        private TranslatorInterface         $translator,
        private MailerInterface             $mailer
    )
    {
    }

    public function __invoke(ResendVerification $resendVerification): void
    {
        $user = $this->userRepository->findByEmail($resendVerification->getEmail()) ?? throw new NotFoundHttpException();

        $verificationToken = $user->getVerificationToken();

        if (empty($verificationToken)) {
            throw new ManualConstraintViolationException('email', 'Account is already verified');
        }

        $token = $user->getVerificationToken();
        $link = $this->portalUrlGenerator->generateUserVerificationUrl($token);

        $templatedEmail = (new TemplatedEmail())
            ->to($user->getEmail())
            ->subject($this->translator->trans('Verification email'))
            ->htmlTemplate('email/user_verification.html.twig')
            ->context([
                'name' => $user->getFirstName(),
                'link' => $link
            ]);
        $this->mailer->send($templatedEmail);
    }
}
