<?php

namespace App\Controller\App;

use App\Entity\Education\Apply;
use App\Entity\System\Message;
use App\Form\Education\App\ApplyType;
use App\Form\Education\App\MessageType;
use App\Repository\Education\ApplyRepository;
use App\Repository\Education\Lesson\CategoryRepository;
use App\Repository\System\Page\PageRepository;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bridge\Twig\Mime\TemplatedEmail;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;

final class AppController extends AbstractController
{
    public function __construct(
        private readonly EntityManagerInterface $entityManager,
        private readonly PageRepository         $pageRepository,
        private readonly CategoryRepository     $categoryRepository,
        private readonly ApplyRepository        $applyRepository,
        private readonly MailerInterface        $mailer
    )
    {
    }

    #[Route('/', name: 'index')]
    public function index(Request $request): Response
    {
        if ($request->headers->get('referer') == 'cms_login' || $request->headers->get('referer') == 'cms_index') {
            return $this->redirectToRoute('cms_index');
        }


        return $this->render('portal/pages/index.html.twig', [
            'page' => $this->pageRepository->findOneBySlug($request->getLocale(), '/')
        ]);


    }

    #[Route('/message', name: 'message')]
    public function message(Request $request): Response
    {
        //addflash
        $message = new Message();

        $form = $this->createForm(MessageType::class, $message, []);

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            $this->entityManager->persist($message);
            $this->entityManager->flush();
            $this->addFlash('success', ['title' => 'message_sending_was_successful', 'text' => 'message_sending_was_successful_text']);
        }

        return $this->redirectToRoute('app_index');
    }

    #[Route('/apply', name: 'apply')]
    public function apply(Request $request): Response
    {
        $apply = new Apply();

        $form = $this->createForm(ApplyType::class, $apply, []);

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            $existApply = $this->applyRepository->findOneByEmail($apply->getEmail());
            if (!$existApply) {
                $this->entityManager->persist($apply);
                $this->entityManager->flush();

                $this->addFlash('success', ['title' => 'apply_was_successful', 'text' => 'apply_was_successful_text']);


                $email = (new TemplatedEmail())
                    ->to($apply->getEmail())
                    ->subject('Feliratkozás megerősítése!')
                    ->htmlTemplate('portal/email/apply_verification.html.twig')
                    ->context([
                        'name' => $apply->getFirstName(),
                        'link' => $this->generateUrl('app_apply_verification', ['apply' => $apply->getId()],UrlGeneratorInterface::ABSOLUTE_URL)

                    ]);

                $this->mailer->send($email);

            } else {
                $this->addFlash('warning', ['title' => 'email_already_in_use', 'text' => 'email_already_in_use_text']);
            }

        }

        return $this->redirectToRoute('app_index');
    }

    #[Route('/apply/verify/{apply}', name: 'apply_verification')]
    public function applyVerify(Request $request, Apply $apply): Response
    {
        if(!$apply->getIsVerified()) {
            $apply->setIsVerified(true);
            $this->entityManager->flush();

            $this->addFlash('success', ['title' => 'apply_verification_was_successful', 'text' => 'apply_verification_was_successful_text']);

            //todo email adminnak


        }else {
            $this->addFlash('warning', ['title' => 'email_already_is_verified', 'text' => 'email_already_is_verified_text']);
        }

        return $this->redirectToRoute('app_index');
    }

    #[Route('/{page}', name: 'page', priority: -100)]
    public function page(Request $request, string $page): Response
    {
        return $this->render('portal/pages/index.html.twig', [
            'page' => $this->pageRepository->findOneBySlug($request->getLocale(), $page)
        ]);
    }


    public function headerRenderer(Request $request)
    {
        return $this->render('portal/components/_header.html.twig', [
            'categories' => $this->categoryRepository->findPublics($request->getLocale()),
            'route_name' => $request->get('_route')
        ]);
    }

    public function footerRenderer(Request $request)
    {
        $message = new Message();

        $form = $this->createForm(MessageType::class, $message, [
            'action' => $this->generateUrl('app_message')
        ]);

        return $this->render('portal/components/_footer.html.twig', [
            'form' => $form
        ]);
    }
}
