<?php

declare(strict_types=1);

namespace App\Controller\Cms\Abstract;

use AllowDynamicProperties;
use App\Transformer\DTO\Cms\ActivityTransformer;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\Persistence\ManagerRegistry;
use Doctrine\Persistence\ObjectRepository;
use Gedmo\Loggable\Entity\LogEntry;
use Omines\DataTablesBundle\DataTableFactory;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Serializer\SerializerInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

#[AllowDynamicProperties]
abstract class BaseController extends AbstractController
{
    public const KEY = '';

    public const PREFIX = '';

    public const ENTITY = '';

    public const FORM = '';

    public const TABLE = '';
    public const LIST_TWIG_PATH = 'system/common/list.html.twig';
    public const MANAGE_TWIG_PATH = 'system/common/manage.html.twig';
    public const ROLES = ['global' => 'ROLE_ADMIN'];
    public string $name;
    public string $prefix;
    public string $entity;
    public string $form;
    public string $table;
    public string $listTwigPath;
    public string $manageTwigPath;
    public array $roles;
    public ObjectRepository $repository;

    public function __construct(
        private readonly DataTableFactory      $dataTableFactory,
        public readonly ManagerRegistry        $managerRegistry,
        private readonly TranslatorInterface   $translator,
        public readonly EntityManagerInterface $entityManager,
        private readonly ActivityTransformer   $activityTransformer,
        public readonly SerializerInterface    $serializer

    )
    {
        $this->name = $this->translator->trans(static::KEY, [], 'cms');
        $this->prefix = 'cms_' . static::PREFIX;
        $this->entity = static::ENTITY;
        $this->form = static::FORM;
        $this->table = static::TABLE;
        $this->listTwigPath = static::LIST_TWIG_PATH;
        $this->manageTwigPath = static::MANAGE_TWIG_PATH;
        $this->roles = static::ROLES;
        $this->repository = $this->managerRegistry->getRepository($this->entity);

        $filterCollection = $this->entityManager->getFilters();
        $filterCollection->disable('softdeleteable');
    }

    public function list(Request $request): Response
    {
        $this->denyAccessUnlessGranted($this->roles['global']);

        $dataTable = $this->dataTableFactory->createFromType($this->table, ['entity' => $this->entity, 'prefix' => $this->prefix, 'locale' => $request->getLocale()])->handleRequest($request);

        if ($dataTable->isCallback()) {
            return $dataTable->getResponse();
        }

        return $this->render($this->listTwigPath, [
            'datatable' => $dataTable,
            'meta' => ['route_prefix' => $this->prefix, 'name' => $this->name],
            'entity' => $this->entity
        ]);
    }

    public function create(Request $request): Response
    {
        $this->denyAccessUnlessGranted($this->roles['global']);

        $entity = new $this->entity();
        $form = $this->createForm($this->form, $entity, ['locale' => $request->getLocale()]);

        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $this->repository->add($entity, true);
            $this->addFlash('success', ['title' => 'creating_was_successful','event' => 'create']);
            return $this->redirectToRoute($this->prefix . 'edit', ['id' => $entity->getId()], Response::HTTP_SEE_OTHER);
        }

        return $this->render($this->manageTwigPath, [
            'entity' => $entity,
            'meta' => ['name' => $this->name, 'route_prefix' => $this->prefix],
            'form' => $form,
        ]);
    }

    public function edit(Request $request, mixed $id): Response
    {
        $this->denyAccessUnlessGranted($this->roles['global']);

        $entity = $this->repository->findOneById($id);
        $form = $this->createForm($this->form, $entity, ['locale' => $request->getLocale()]);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            $this->repository->add($entity, true);
            $this->addFlash('success', ['title' => 'editing_was_successful','event' => 'edit']);
            return $this->redirectToRoute($this->prefix . 'edit', ['id' => $entity->getId()], Response::HTTP_SEE_OTHER);
        }

        return $this->render($this->manageTwigPath, [
            'entity' => $entity,
            'meta' => ['name' => $this->name, 'route_prefix' => $this->prefix],
            'form' => $form,
            'activites' => array_map($this->activityTransformer, $this->entityManager->getRepository(LogEntry::class)->getLogEntries($entity)),
        ]);
    }

    public function delete(Request $request, mixed $id): Response
    {
        $this->denyAccessUnlessGranted($this->roles['global']);

        $entity = $this->repository->findOneById($id);
        if ($this->isCsrfTokenValid('delete' . $entity->getId(), $request->request->get('_token'))) {
            $this->repository->remove($entity, true);
        }

        $this->addFlash('success', ['title' => 'deleting_was_successful','event' => 'delete']);
        return $this->redirectToRoute($this->prefix . 'list', [], Response::HTTP_SEE_OTHER);
    }


    public function restore(Request $request, mixed $id): Response
    {
        $this->denyAccessUnlessGranted($this->roles['global']);

        $entity = $this->repository->findOneById($id);
        if ($this->isCsrfTokenValid('restore' . $entity->getId(), $request->request->get('_token'))) {
            $this->repository->restore($entity, true);
            $this->addFlash('success', ['title' => 'restoring_was_successful','event' => 'restore']);
        }

        return $this->redirectToRoute($this->prefix . 'list', [], Response::HTTP_SEE_OTHER);
    }


    public function tree(): Response
    {
        $this->denyAccessUnlessGranted($this->roles['global']);
        $entity = $this->repository->findOneByParent(null);
        return $this->render('system/common/tree.html.twig', [
            'meta' => ['route_prefix' => $this->prefix, 'name' => $this->name],
            'entity' => $entity
        ]);
    }

    public function treeApi(Request $request, $nestedTransformer): JsonResponse
    {
        $this->denyAccessUnlessGranted($this->roles['global']);

        $entity = $this->repository->createQueryBuilder('n')->andWhere('n.parent IS NULL')->orderBy('n.position', 'ASC')->getQuery()->getResult();
        if ($request->isMethod('POST')) {

            $data = $request->getContent();
            $data = json_decode($data, true, 512, JSON_THROW_ON_ERROR);

            $this->updateTree($data);
        }

        return new JsonResponse($this->serializer->normalize(array_map($nestedTransformer, $entity)));
    }

    public function updateTree(array $nodes, $parent = null): void
    {
        $em = $this->entityManager;
        $entityRepository = $em->getRepository($this->entity);

        foreach ($nodes as $index => $node) {
            $entity = $entityRepository->findOneById($node['id']);

            if ($entity) {
                $entity->setPosition($index);

                $parent = $parent ? $entityRepository->findOneById($parent) : $parent;
                $entity->setParent($parent);

                $em->persist($entity);

                if (isset($node['children']) && !empty($node['children'])) {
                    $this->updateTree($node['children'], $entity->getId());
                }
            }

            $entityRepository->reorder($entity, 'position', 'ASC', false);
        }

        $em->flush();
    }

}
