<?php

declare(strict_types=1);

namespace App\Controller\Cms;

use App\Repository\Education\ApplyRepository;
use App\Service\Chart\UserChart;
use Doctrine\ORM\EntityManagerInterface;
use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xls;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use Scheb\TwoFactorBundle\Security\TwoFactor\Provider\Google\GoogleAuthenticatorInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Annotation\Route;

final class ExportController extends AbstractController
{
    public function __construct(
        private readonly EntityManagerInterface $entityManager,
        private readonly ApplyRepository        $applyRepository
    )
    {
    }

    #[Route('/export/unhandled', name: 'export_unhandled')]
    public function index(): Response
    {
        $this->denyAccessUnlessGranted('ROLE_ADMIN');
        $applies = $this->applyRepository->collectForExport();

        $lines = [];

        foreach ($applies as $apply) {
            $lines[] = $apply->getEmail();
            $apply->setExportedAt(new \DateTimeImmutable());
        }
        $this->entityManager->flush();

        $data = implode(";", $lines);

        $response = new Response($data);
        $response->headers->set('Content-Type', 'text/plain');
        $response->headers->set('Content-Disposition', 'attachment; filename="export.txt"');

        return $response;
    }

    #[Route('/export/date', name: 'export_date')]
    public function date(Request $request): Response
    {
        $this->denyAccessUnlessGranted('ROLE_ADMIN');


        $dateFrom = $request->get('dateFrom');
        $dateTo = $request->get('dateTo');

        $applies = $this->applyRepository->findByDate($dateFrom, $dateTo);

        $lines = [];

        foreach ($applies as $apply) {
            $lines[] = [$apply->getLastName(), $apply->getFirstName(), $apply->getEmail(), $apply->getExportedAt() ? $apply->getExportedAt()->format('Y. m. d. H:i') : ''];
        }

        $csvData = $this->arrayToCsv($lines);

        $spreadsheet = new Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();

        $sheet->fromArray($lines, NULL, 'A1');

        header('Content-Disposition: attachment;filename="export_date.xls"');
        header('Cache-Control: max-age=0');

        $writer = new Xls($spreadsheet);
        $writer->save('php://output');
        return new Response();
    }

    private function arrayToCsv(array $data): string
    {
        $output = fopen('php://temp', 'w');
        foreach ($data as $row) {
            fputcsv($output, $row, ';');
        }
        rewind($output);
        $csvContent = stream_get_contents($output);
        fclose($output);
        return $csvContent;
    }
}
