<?php

declare(strict_types=1);

namespace App\Controller\Cms\Security;

use App\Entity\System\Administrator\Administrator;
use App\Form\System\ChangePasswordFormType;
use App\Form\System\ResetPasswordRequestFormType;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bridge\Twig\Mime\TemplatedEmail;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Contracts\Translation\TranslatorInterface;
use SymfonyCasts\Bundle\ResetPassword\Controller\ResetPasswordControllerTrait;
use SymfonyCasts\Bundle\ResetPassword\Exception\ResetPasswordExceptionInterface;
use SymfonyCasts\Bundle\ResetPassword\Model\ResetPasswordToken;
use SymfonyCasts\Bundle\ResetPassword\ResetPasswordHelperInterface;

#[Route('/reset-password', name: 'reset_password_')]
final class ResetPasswordController extends AbstractController
{
    use ResetPasswordControllerTrait;

    public function __construct(
        private readonly ResetPasswordHelperInterface $resetPasswordHelper,
        private readonly EntityManagerInterface $entityManager
    ) {
    }


    #[Route('', name: 'request')]
    public function request(Request $request, MailerInterface $mailer, TranslatorInterface $translator): Response
    {
        $form = $this->createForm(ResetPasswordRequestFormType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {
            return $this->processSendingPasswordResetEmail(
                $form->get('email')->getData(),
                $mailer
            );
        }

        return $this->render('system/security/reset/request.html.twig', [
            'requestForm' => $form,
        ]);
    }
    
    #[Route('/check-email', name: 'check_email')]
    public function checkEmail(): Response
    {

        if (!($resetToken = $this->getTokenObjectFromSession()) instanceof ResetPasswordToken) {
            $resetToken = $this->resetPasswordHelper->generateFakeResetToken();
        }

        return $this->render('system/security/reset/check_email.html.twig', [
            'resetToken' => $resetToken,
        ]);
    }


    #[Route('/reset/{token}', name: 'reset')]
    public function reset(
        Request $request,
        UserPasswordHasherInterface $userPasswordHasher,
        TranslatorInterface $translator,
        string $token = null
    ): Response {
        if ($token) {
            $this->storeTokenInSession($token);

            return $this->redirectToRoute('cms_reset_password_reset');
        }

        $token = $this->getTokenFromSession();
        if (null === $token) {
            throw $this->createNotFoundException('No reset password token found in the URL or in the session.');
        }

        try {
            $user = $this->resetPasswordHelper->validateTokenAndFetchUser($token);
        } catch (ResetPasswordExceptionInterface $resetPasswordException) {
            $this->addFlash('reset_password_error', sprintf(
                '%s - %s',
                $translator->trans(ResetPasswordExceptionInterface::MESSAGE_PROBLEM_VALIDATE, [], 'ResetPasswordBundle'),
                $translator->trans($resetPasswordException->getReason(), [], 'ResetPasswordBundle')
            ));

            return $this->redirectToRoute('cms_reset_password_request');
        }

        $form = $this->createForm(ChangePasswordFormType::class);
        $form->handleRequest($request);

        if ($form->isSubmitted() && $form->isValid()) {

            $this->resetPasswordHelper->removeResetRequest($token);

            $encodedPassword = $userPasswordHasher->hashPassword(
                $user,
                $form->get('plainPassword')->getData()
            );

            $user->setPassword($encodedPassword);
            $this->entityManager->flush();

            $this->cleanSessionAfterReset();

            return $this->redirectToRoute('cms_reset_password_finish');
        }

        return $this->render('system/security/reset/reset.html.twig', [
            'resetForm' => $form,
        ]);
    }

    private function processSendingPasswordResetEmail(string $emailFormData, MailerInterface $mailer): RedirectResponse
    {
        $administrator = $this->entityManager->getRepository(Administrator::class)->findOneBy([
            'email' => $emailFormData,
        ]);


        if (!$administrator instanceof Administrator) {
            return $this->redirectToRoute('cms_reset_password_check_email');
        }

        try {
            $resetPasswordToken = $this->resetPasswordHelper->generateResetToken($administrator);
        } catch (ResetPasswordExceptionInterface) {
            return $this->redirectToRoute('cms_reset_password_check_email');
        }

        $templatedEmail = (new TemplatedEmail())
            ->to($administrator->getEmail())
            ->subject('Your password reset request')
            ->htmlTemplate('system/security/reset/email.html.twig')
            ->context([
                'resetToken' => $resetPasswordToken,
            ])
        ;

        $mailer->send($templatedEmail);

        $this->setTokenObjectInSession($resetPasswordToken);

        return $this->redirectToRoute('cms_reset_password_check_email');
    }

    #[Route('/finish', name: 'finish')]
    public function index(): Response
    {
        return $this->render('system/security/reset/finish.html.twig');
    }
}
