<?php

declare(strict_types=1);

namespace App\Controller\Portal\Auth\Registration;

use App\DTO\PublicAPI\User\Input\FormRegistration;
use App\Enum\User\RegistrationSource;
use App\Event\User\UserRegisteredEvent;
use App\Exception\View\ConstraintViolationException;
use App\Factory\User\UserFactory;
use App\Repository\User\UserRepository;
use App\Service\View\ViewHandlerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Attribute\MapRequestPayload;
use Symfony\Component\Routing\Annotation\Route;
use Symfony\Component\Validator\Validator\ValidatorInterface;

#[Route(path: '/auth/registration/register', name: 'auth_registration_register', methods: ['POST'])]
final class RegisterController extends AbstractController
{
    public function __construct(
        private readonly ValidatorInterface       $validator,
        private readonly UserFactory              $userFactory,
        private readonly UserRepository           $userRepository,
        private readonly EventDispatcherInterface $eventDispatcher,
        private readonly ViewHandlerInterface     $viewHandler
    )
    {
    }

    public function __invoke(#[MapRequestPayload] FormRegistration $formRegistration, Request $request): Response
    {
        $user = $this->userFactory->createFromRegistration($formRegistration,$request->getLocale());

        $violations = $this->validator->validate($user);

        if ($violations->count() > 0) {
            throw new ConstraintViolationException($violations);
        }

        $this->userRepository->add($user, true);

        $this->eventDispatcher->dispatch(new UserRegisteredEvent($user, RegistrationSource::RegistrationForm));

        return $this->viewHandler->handle([]);
    }
}
