<?php

declare(strict_types=1);

namespace App\Controller\Portal\System;

use App\DTO\PublicAPI\Address\Input\AddressManage;
use App\Entity\Common\Address;
use App\Factory\Shared\AddressFactory;
use App\Service\View\ViewHandlerInterface;
use App\Transformer\DTO\Shared\AddressTransformer;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Attribute\MapRequestPayload;
use Symfony\Component\Routing\Annotation\Route;

#[Route(path: '/system/addresses', name: 'system_address_')]
final class AddressController extends AbstractController
{
    public function __construct(
        private readonly EntityManagerInterface $entityManager,
        private readonly AddressTransformer     $addressTransformer,
        private readonly AddressFactory         $addressFactory,
        private readonly ViewHandlerInterface   $viewHandler,
    )
    {
    }

    #[Route(path: '', name: 'create', methods: ['POST'])]
    public function createAction(#[MapRequestPayload] AddressManage $addressManage): Response
    {
        return $this->manage($addressManage);
    }

    #[Route(path: '/{address}', name: 'update', methods: ['PATCH'])]
    public function updateAction(#[MapRequestPayload] AddressManage $addressManage, Address $address): Response
    {
        return $this->manage($addressManage, $address);
    }

    private function manage(AddressManage $addressManage, ?Address $address = null): Response
    {
        $address = $this->addressFactory->manage($addressManage, $address);
        $this->entityManager->persist($address);

        $this->entityManager->flush();

        $transformer = $this->addressTransformer;
        $address = $transformer($address);

        return $this->viewHandler->handle($address);
    }

    #[Route(path: '/{address}', name: 'view', methods: ['GET'])]
    public function viewAction(Address $address): Response
    {
        $transformer = $this->addressTransformer;
        $address = $transformer($address);

        return $this->viewHandler->handle($address);
    }
}
