<?php

namespace App\DataFixtures;

use AllowDynamicProperties;
use App\Entity\System\Administrator\Administrator;
use App\Entity\User\User;
use App\Enum\Shared\TwoFactorProvider;
use Doctrine\Bundle\FixturesBundle\Fixture;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\Persistence\ObjectManager;
use Scheb\TwoFactorBundle\Security\TwoFactor\Provider\Google\GoogleAuthenticatorInterface;
use Symfony\Component\DependencyInjection\ParameterBag\ParameterBagInterface;
use Symfony\Component\PasswordHasher\Hasher\UserPasswordHasherInterface;

#[AllowDynamicProperties]
final class AppFixtures extends Fixture
{
    final public const ADMIN_REFERENCE = 'admin-reference';

    /**
     * @var string
     */
    private const PLAINTEXT_PASSWORD = 'simple';

    public function __construct(
        private readonly EntityManagerInterface       $entityManager,
        private readonly UserPasswordHasherInterface  $userPasswordHasher,
        private readonly GoogleAuthenticatorInterface $googleAuthenticator,
        private readonly ParameterBagInterface        $parameterBag
    )
    {
    }

    public function load(ObjectManager $manager): void
    {
        $this->createAdministrator();
        $this->createPortalUser();
        $manager->flush();
    }

    private function createAdministrator(): void
    {
        $administrator = new Administrator();
        $administrator->setEmail('admin@beetdev.hu');
        $administrator->setFirstName('Admin');
        $administrator->setLastName('Super');
        $administrator->setLocale($this->parameterBag->get('default_locale'));
        $hashedPassword = $this->userPasswordHasher->hashPassword($administrator, self::PLAINTEXT_PASSWORD);
        $administrator->setPassword($hashedPassword);
        $administrator->setRoles(['ROLE_SUPER_ADMIN']);
        $administrator->setGoogleAuthenticatorSecret($this->googleAuthenticator->generateSecret());
        $administrator->setTwoFactorProvider(TwoFactorProvider::EMAIL);

        $this->entityManager->persist($administrator);

        $this->addReference(self::ADMIN_REFERENCE, $administrator);
    }

    private function createPortalUser(): void
    {
        $user = new User();
        $user->setFirstName('Portal ');
        $user->setLastName('User');
        $user->setEmail('portal.user@dev.simple-cms.hu');
        $user->setLocale($this->parameterBag->get('default_locale'));
        $user->setVerified(true);

        $password = $this->userPasswordHasher->hashPassword(
            $user, self::PLAINTEXT_PASSWORD
        );
        $user->setPassword($password);

        $this->entityManager->persist($user);
    }
}
