<?php

declare(strict_types=1);

namespace App\Entity\Education\Lesson;

use App\Entity\Education\Apply;
use App\Entity\Interfaces\BlockableInterface;
use App\Entity\Interfaces\MetableInterface;
use App\Entity\Interfaces\TagableInterface;
use App\Entity\Shared\AbstractTranslatableEntity;
use App\Repository\Education\Lesson\CategoryRepository;
use App\Traits\Entity\Blockable;
use App\Traits\Entity\ImageAware;
use App\Traits\Entity\Metable;
use App\Traits\Entity\Tagable;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;

#[ORM\Table('education_lesson_category')]
#[ORM\Entity(repositoryClass: CategoryRepository::class)]
class Category extends AbstractTranslatableEntity implements MetableInterface, TagableInterface, BlockableInterface
{
    use Metable, Tagable, Blockable, ImageAware;

    #[ORM\OneToMany(mappedBy: 'category', targetEntity: Lesson::class)]
    #[ORM\OrderBy(['sort' => 'ASC'])]
    private Collection $lessons;

    #[ORM\ManyToMany(targetEntity: Apply::class, mappedBy: 'categories')]
    private Collection $applies;

    #[ORM\Column]
    private ?int $sort = 0;

    #[ORM\Column(options: ['default' => false])]
    private ?bool $isApply = false;

    #[ORM\Column(length: 255, nullable: true)]
    private ?string $highlightedImage = null;

    #[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
    private ?\DateTimeInterface $startDate = null;

    #[ORM\Column(type: Types::DATETIME_MUTABLE, nullable: true)]
    private ?\DateTimeInterface $endDate = null;

    public function __construct()
    {
        $this->lessons = new ArrayCollection();
        $this->applies = new ArrayCollection();
    }

    /**
     * @return Collection<int, Lesson>
     */
    public function getLessons(): Collection
    {
        return $this->lessons;
    }

    public function addLesson(Lesson $lesson): static
    {
        if (!$this->lessons->contains($lesson)) {
            $this->lessons->add($lesson);
            $lesson->setCategory($this);
        }

        return $this;
    }

    public function removeLesson(Lesson $lesson): static
    {
        if ($this->lessons->removeElement($lesson)) {
            // set the owning side to null (unless already changed)
            if ($lesson->getCategory() === $this) {
                $lesson->setCategory(null);
            }
        }

        return $this;
    }

    /**
     * @return Collection<int, Apply>
     */
    public function getApplies(): Collection
    {
        return $this->applies;
    }

    public function addApply(Apply $apply): static
    {
        if (!$this->applies->contains($apply)) {
            $this->applies->add($apply);
            $apply->addCategory($this);
        }

        return $this;
    }

    public function removeApply(Apply $apply): static
    {
        if ($this->applies->removeElement($apply)) {
            $apply->removeCategory($this);
        }

        return $this;
    }

    public function getSort(): ?int
    {
        return $this->sort;
    }

    public function setSort(int $sort): static
    {
        $this->sort = $sort;

        return $this;
    }

    public function isIsApply(): ?bool
    {
        return $this->isApply;
    }

    public function setIsApply(bool $isApply): static
    {
        $this->isApply = $isApply;

        return $this;
    }

    public function getHighlightedImage(): ?string
    {
        return $this->highlightedImage;
    }

    public function setHighlightedImage(?string $highlightedImage): static
    {
        $this->highlightedImage = $highlightedImage;

        return $this;
    }

    public function getStartDate(): ?\DateTimeInterface
    {
        return $this->startDate;
    }

    public function setStartDate(?\DateTimeInterface $startDate): static
    {
        $this->startDate = $startDate;

        return $this;
    }

    public function getEndDate(): ?\DateTimeInterface
    {
        return $this->endDate;
    }

    public function setEndDate(?\DateTimeInterface $endDate): static
    {
        $this->endDate = $endDate;

        return $this;
    }
}
