<?php

declare(strict_types=1);

namespace App\Entity\Magazine\Article;

use App\Entity\Interfaces\BlockableInterface;
use App\Entity\Interfaces\MetableInterface;
use App\Entity\Interfaces\TagableInterface;
use App\Entity\Shared\AbstractTranslatableEntity;
use App\Entity\System\Administrator\Administrator;
use App\Repository\Magazine\Article\ArticleRepository;
use App\Traits\Entity\Blockable;
use App\Traits\Entity\ImageAware;
use App\Traits\Entity\Metable;
use App\Traits\Entity\Tagable;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;

#[ORM\Table('magazine_article')]
#[ORM\Entity(repositoryClass: ArticleRepository::class)]
class Article extends AbstractTranslatableEntity implements MetableInterface, TagableInterface, BlockableInterface
{
    use Metable, Tagable, Blockable, ImageAware;

    #[ORM\ManyToOne(inversedBy: 'magazineArticles')]
    #[ORM\JoinColumn(nullable: false)]
    private ?Administrator $author = null;

    #[ORM\ManyToMany(targetEntity: self::class, inversedBy: 'articles')]
    #[ORM\JoinTable('magazine_article_to_related_article')]
    private Collection $relatedArticles;

    #[ORM\ManyToMany(targetEntity: self::class, mappedBy: 'relatedArticles')]
    private Collection $articles;

    public function __construct()
    {
        $this->relatedArticles = new ArrayCollection();
        $this->articles = new ArrayCollection();
    }

    public function getAuthor(): ?Administrator
    {
        return $this->author;
    }

    public function setAuthor(?Administrator $author): self
    {
        $this->author = $author;

        return $this;
    }

    public function getRelatedArticles(): Collection
    {
        return $this->relatedArticles;
    }

    public function getArticles(): Collection
    {
        return $this->articles;
    }

    public function addArticle(self $article): self
    {
        if (!$this->articles->contains($article)) {
            $this->articles->add($article);
            $article->addRelatedArticle($this);
        }

        return $this;
    }

    public function addRelatedArticle(self $relatedArticle): self
    {
        if (!$this->relatedArticles->contains($relatedArticle)) {
            $this->relatedArticles->add($relatedArticle);
        }

        return $this;
    }

    public function removeArticle(self $article): self
    {
        if ($this->articles->removeElement($article)) {
            $article->removeRelatedArticle($this);
        }

        return $this;
    }

    public function removeRelatedArticle(self $relatedArticle): self
    {
        $this->relatedArticles->removeElement($relatedArticle);

        return $this;
    }
}
