<?php

declare(strict_types=1);

namespace App\Entity\System\Administrator;


use App\Entity\Magazine\Article\Article;
use App\Entity\Shared\AbstractEntity;
use App\Enum\Shared\TwoFactorProvider;
use App\Repository\System\Administrator\AdministratorRepository;
use App\Traits\Entity\NameAware;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;
use Doctrine\DBAL\Types\Types;
use Doctrine\ORM\Mapping as ORM;
use Gedmo\Mapping\Annotation as Gedmo;
use Scheb\TwoFactorBundle\Model\BackupCodeInterface;
use Scheb\TwoFactorBundle\Model\Email\TwoFactorInterface as EmailTwoFactorInterface;
use Scheb\TwoFactorBundle\Model\Google\TwoFactorInterface as GoogleTwoFactorInterface;
use Scheb\TwoFactorBundle\Model\PreferredProviderInterface;
use Symfony\Component\Security\Core\User\PasswordAuthenticatedUserInterface;
use Symfony\Component\Security\Core\User\UserInterface;

#[ORM\Entity(repositoryClass: AdministratorRepository::class)]
class Administrator extends AbstractEntity implements UserInterface, PasswordAuthenticatedUserInterface, GoogleTwoFactorInterface, EmailTwoFactorInterface, BackupCodeInterface, PreferredProviderInterface
{
    use NameAware;

    private const BACKUP_CODES = [111111];

    #[ORM\Column(length: 180, unique: true)]
    #[Gedmo\Versioned]
    private ?string $email = null;

    #[ORM\Column(length: 360, nullable: true)]
    private ?string $googleAuthenticatorSecret = null;

    #[ORM\Column]
    private array $roles = [];

    #[ORM\Column]
    private ?string $password = null;

    #[ORM\Column(type: Types::TEXT, nullable: true)]
    private ?string $agent = null;

    #[ORM\Column(length: 3)]
    #[Gedmo\Versioned]
    private ?string $locale = null;

    #[ORM\Column(length: 255, nullable: true)]
    private ?string $authCode = null;

    #[ORM\OneToMany(mappedBy: 'author', targetEntity: Article::class)]
    private Collection $magazineArticles;

    #[ORM\Column(length: 50, enumType: TwoFactorProvider::class, options: ['default' => TwoFactorProvider::EMAIL])]
    #[Gedmo\Versioned]
    private ?TwoFactorProvider $twoFactorProvider = null;

    public function __construct()
    {
        $this->magazineArticles = new ArrayCollection();
    }

    public function getEmail(): ?string
    {
        return $this->email;
    }

    public function setEmail(string $email): self
    {
        $this->email = $email;

        return $this;
    }

    /**
     * A visual identifier that represents this user.
     *
     * @see UserInterface
     */
    public function getUserIdentifier(): string
    {
        return (string)$this->email;
    }

    /**
     * @see UserInterface
     */
    public function getRoles(): array
    {
        $roles = $this->roles;
        // guarantee every user at least has ROLE_USER
        $roles[] = 'ROLE_USER';

        return array_unique($roles);
    }

    public function setRoles(array $roles): self
    {
        $this->roles = $roles;

        return $this;
    }

    /**
     * @see PasswordAuthenticatedUserInterface
     */
    public function getPassword(): string
    {
        return $this->password;
    }

    public function setPassword(string $password): self
    {
        $this->password = $password;

        return $this;
    }

    /**
     * @see UserInterface
     */
    public function eraseCredentials(): void
    {
        // If you store any temporary, sensitive data on the user, clear it here
        // $this->plainPassword = null;
    }

    public function isGoogleAuthenticatorEnabled(): bool
    {
        return null !== $this->googleAuthenticatorSecret;
    }

    public function getGoogleAuthenticatorUsername(): string
    {
        return $this->email;
    }

    public function getGoogleAuthenticatorSecret(): ?string
    {
        return $this->googleAuthenticatorSecret;
    }

    public function setGoogleAuthenticatorSecret(?string $googleAuthenticatorSecret): void
    {
        $this->googleAuthenticatorSecret = $googleAuthenticatorSecret;
    }

    public function isBackupCode(string $code): bool
    {
        return \in_array($code, self::BACKUP_CODES);
    }

    public function invalidateBackupCode(string $code): void
    {
    }

    public function isEmailAuthEnabled(): bool
    {
        return true; // This can be a persisted field to switch email code authentication on/off
    }

    public function getEmailAuthRecipient(): string
    {
        return $this->email;
    }

    public function getEmailAuthCode(): string
    {
        return $this->authCode;
    }

    public function setEmailAuthCode(string $authCode): void
    {
        $this->authCode = $authCode;
    }

    public function getAgent(): ?string
    {
        return $this->agent;
    }

    public function setAgent(?string $agent): self
    {
        $this->agent = $agent;

        return $this;
    }

    public function getLocale(): ?string
    {
        return $this->locale;
    }

    public function setLocale(string $locale): self
    {
        $this->locale = $locale;

        return $this;
    }

    /**
     * @return Collection<int, Article>
     */
    public function getMagazineArticles(): Collection
    {
        return $this->magazineArticles;
    }

    public function addMagazineArticle(Article $magazineArticle): self
    {
        if (!$this->magazineArticles->contains($magazineArticle)) {
            $this->magazineArticles->add($magazineArticle);
            $magazineArticle->setAuthor($this);
        }

        return $this;
    }

    public function removeMagazineArticle(Article $magazineArticle): self
    {
        if ($this->magazineArticles->removeElement($magazineArticle) && $magazineArticle->getAuthor() === $this) {
            $magazineArticle->setAuthor(null);
        }

        return $this;
    }

    public function provideUser()
    {
        return $this->id;
    }

    public function provideUserEntity(): ?string
    {
        return $this->id;
    }


    public function getPreferredTwoFactorProvider(): ?string
    {
        return $this->twoFactorProvider->value;
    }

    public function getTwoFactorProvider(): ?TwoFactorProvider
    {
        return $this->twoFactorProvider;
    }

    public function setTwoFactorProvider(TwoFactorProvider $twoFactorProvider): self
    {
        $this->twoFactorProvider = $twoFactorProvider;

        return $this;
    }
}
