<?php

declare(strict_types=1);

namespace App\EventSubscriber\Portal\Email;

use App\Event\User\UserPasswordChangedEvent;
use Symfony\Bridge\Twig\Mime\TemplatedEmail;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

final readonly class UserPasswordChangedEmailEventSubscriber implements EventSubscriberInterface
{
    public function __construct(
        private TranslatorInterface $translator,
        private MailerInterface     $mailer
    )
    {
    }

    public static function getSubscribedEvents(): array
    {
        return [
            UserPasswordChangedEvent::class => 'handle'
        ];
    }

    public function handle(UserPasswordChangedEvent $userPasswordChangedEvent): void
    {
        if (!$userPasswordChangedEvent->shouldSendEmail()) {
            return;
        }

        $templatedEmail = (new TemplatedEmail())
            ->to($userPasswordChangedEvent->getUser()->getEmail())
            ->subject($this->translator->trans('Successful password change'))
            ->htmlTemplate('email/user_password_changed.html.twig')
            ->context([
                'name' => $userPasswordChangedEvent->getUser()->getFirstName()
            ]);
        $this->mailer->send($templatedEmail);
    }
}
