<?php

declare(strict_types=1);

namespace App\EventSubscriber\Portal\Email;

use App\Entity\User\User;
use App\Event\User\UserRegisteredEvent;
use App\Repository\User\UserRepository;
use App\Service\Authentication\Verification\Generator\VerificationTokenGeneratorInterface;
use App\Service\Url\PortalUrlGeneratorInterface;
use Symfony\Bridge\Twig\Mime\TemplatedEmail;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\Mailer\MailerInterface;
use Symfony\Contracts\Translation\TranslatorInterface;

final readonly class UserVerificationEmailEventSubscriber implements EventSubscriberInterface
{
    public function __construct(
        private VerificationTokenGeneratorInterface $verificationTokenGenerator,
        private UserRepository                      $userRepository,
        private PortalUrlGeneratorInterface         $portalUrlGenerator,
        private TranslatorInterface                 $translator,
        private MailerInterface                     $mailer
    )
    {
    }

    public static function getSubscribedEvents(): array
    {
        return [
            UserRegisteredEvent::class => 'handle'
        ];
    }

    public function handle(UserRegisteredEvent $userRegisteredEvent): void
    {
        $user = $userRegisteredEvent->getUser();

        if ($user->isVerified()) {
            return;
        }

        $verificationToken = $this->verificationTokenGenerator->generate();
        $token = $verificationToken->getToken();
        $user->setVerificationToken($token);

        $this->userRepository->add($user, true);

        $this->send($user);
    }

    private function send(User $user): void
    {
        $verificationToken = $user->getVerificationToken();
        $link = $this->portalUrlGenerator->generateUserVerificationUrl($verificationToken);

        $templatedEmail = (new TemplatedEmail())
            ->to($user->getEmail())
            ->subject($this->translator->trans('Verification email'))
            ->htmlTemplate('email/user_verification.html.twig')
            ->context([
                'name' => $user->getFullName(),
                'link' => $link
            ]);
        $this->mailer->send($templatedEmail);
    }
}
