<?php

declare(strict_types=1);

namespace App\EventSubscriber\View;

use App\Service\View\ViewHandlerInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\Event\ExceptionEvent;
use Symfony\Component\Messenger\Exception\HandlerFailedException;
use Throwable;

final class ExceptionEventSubscriber implements EventSubscriberInterface
{
    private const PROXY_EXCEPTIONS = [
        HandlerFailedException::class
    ];

    public function __construct(
        private readonly RequestStack         $requestStack,
        private readonly ViewHandlerInterface $viewHandler
    )
    {
    }

    public static function getSubscribedEvents(): array
    {
        return [
            ExceptionEvent::class => [
                ['onException', -10]
            ]
        ];
    }

    public function onException(ExceptionEvent $exceptionEvent): void
    {
        $request = $this->requestStack->getMainRequest();

        if ($request === null) {
            return;
        }

        $route = $request->attributes->get('_route') ?: '';

        if (!str_starts_with($route, 'portal_api')) {
            return;
        }

        $exception = $exceptionEvent->getThrowable();

        if (in_array(get_class($exception), self::PROXY_EXCEPTIONS)) {
            $previousException = $exception->getPrevious();

            if ($previousException !== null) {
                $exception = $previousException;
            }
        }

        try {
            $response = $this->viewHandler->handle($exception);
        } catch (Throwable $exception) {
            try {
                $response = $this->viewHandler->handle($exception);
            } catch (Throwable) {
                $response = new JsonResponse([], 500);
            }
        } finally {
            $exceptionEvent->setResponse($response);
        }
    }
}
