<?php

declare(strict_types=1);

namespace App\Repository\Common;

use App\DTO\PublicAPI\Shared\Input\FaqFilter;
use App\Entity\Common\Support\Faq;
use App\Repository\Shared\PaginationAwareRepositoryTrait;
use App\Service\Pagination\Context\PaginationContext;
use App\Traits\Repository\Editable;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\Persistence\ManagerRegistry;

final class FaqRepository extends ServiceEntityRepository
{
    use Editable, PaginationAwareRepositoryTrait;

    public function __construct(ManagerRegistry $managerRegistry)
    {
        parent::__construct($managerRegistry, Faq::class);
    }

    public function findPublicFaqs(
        string             $locale,
        ?PaginationContext $paginationContext,
        ?FaqFilter         $faqFilter,

    ): array
    {
        $queryBuilder = $this->createQueryBuilder('t')
            ->addSelect('tt')
            ->leftJoin('t.translations', 'tt')
            ->andWhere('tt.locale = :locale')
            ->andWhere('tt.isActive = true')
            ->andWhere('tt.deletedAt IS NULL')
            ->andWhere('t.deletedAt IS NULL')
            ->orderBy('tt.createdAt', 'ASC')
            ->setParameter('locale', $locale);

        if (!empty($faqFilter?->question)) {
            $queryBuilder
                ->andWhere(
                    $queryBuilder->expr()->like('UPPER(tt.question)', 'UPPER(:question)')
                )
                ->setParameter('question', '%' . $faqFilter->question . '%');
        }

        if (!empty($faqFilter?->answer)) {
            $queryBuilder
                ->andWhere(
                    $queryBuilder->expr()->like('UPPER(tt.answer)', 'UPPER(:answer)')
                )
                ->setParameter('answer', '%' . $faqFilter->answer . '%');
        }

        $this->applyPagination($queryBuilder, $paginationContext);

        return $queryBuilder
            ->getQuery()
            ->getResult();
    }


}
