<?php

declare(strict_types=1);

namespace App\Repository\Common;

use App\DTO\PublicAPI\Shared\Input\TagFilter;
use App\Entity\Common\Tag\Tag;
use App\Repository\Shared\PaginationAwareRepositoryTrait;
use App\Service\Pagination\Context\PaginationContext;
use App\Traits\Repository\Editable;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\Persistence\ManagerRegistry;

final class TagRepository extends ServiceEntityRepository
{
    use Editable, PaginationAwareRepositoryTrait;

    public function __construct(ManagerRegistry $managerRegistry)
    {
        parent::__construct($managerRegistry, Tag::class);
    }

    public function findPublicTags(
        string             $locale,
        ?PaginationContext $paginationContext,
        ?TagFilter         $tagFilter,

    ): array
    {
        $queryBuilder = $this->createQueryBuilder('t')
            ->addSelect('tt')
            ->leftJoin('t.translations', 'tt', 'WITH', 'tt.locale = :locale')
            ->orderBy('tt.createdAt', 'ASC')
            ->andWhere('tt.isActive = true')
            ->andWhere('tt.deletedAt IS NULL')
            ->andWhere('t.deletedAt IS NULL')
            ->setParameter('locale', $locale);

        if (!empty($tagFilter?->keyword)) {
            $queryBuilder
                ->andWhere(
                    $queryBuilder->expr()->like('UPPER(tt.title)', 'UPPER(:keyword)')
                )
                ->setParameter('keyword', '%' . $tagFilter->keyword . '%');
        }

        if (isset($tagFilter?->isHighlighted)) {
            $queryBuilder
                ->andWhere('tt.isHighlighted = :isHighlighted')
                ->setParameter('isHighlighted', $tagFilter->isHighlighted);
        }

        $this->applyPagination($queryBuilder, $paginationContext);

        return $queryBuilder
            ->getQuery()
            ->getResult();
    }


}
