<?php

declare(strict_types=1);

namespace App\Repository\Education\Lesson;

use App\Entity\Education\Lesson\Category;
use App\Repository\Shared\PaginationAwareRepositoryTrait;
use App\Service\Pagination\Context\PaginationContext;
use App\Traits\Repository\Editable;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\Persistence\ManagerRegistry;

final class CategoryRepository extends ServiceEntityRepository
{
    use Editable, PaginationAwareRepositoryTrait;

    public function __construct(ManagerRegistry $managerRegistry)
    {
        parent::__construct($managerRegistry, Category::class);
    }

    public function findPublics($locale, ?PaginationContext $paginationContext = null): array
    {
        $today = new \DateTime('today');

        $queryBuilder = $this->createQueryBuilder('a')
            ->addSelect('at')
            ->leftJoin('a.translations', 'at')
            ->andWhere('at.locale = :locale')
            ->andWhere('at.isActive = true')
            ->addSelect('CASE WHEN :today BETWEEN a.startDate AND a.endDate THEN 0 ELSE 1 END as HIDDEN isCurrent')
            ->orderBy('isCurrent', 'ASC')
            ->addOrderBy('a.startDate', 'ASC')
            ->setParameter('locale', $locale)
            ->setParameter('today', $today);

        $this->applyPagination($queryBuilder, $paginationContext);

        return $queryBuilder
            ->getQuery()
            ->getResult();
    }




    public function findOneBySlug($locale, $slug): ?Category
    {
        return $this->createQueryBuilder('a')
            ->leftJoin('a.translations', 'trans')
            ->andWhere('trans.slug = :slug')
            ->andWhere('trans.locale = :locale')
            ->setMaxResults(1)
            ->setParameter('slug', $slug)
            ->setParameter('locale', $locale)
            ->getQuery()
            ->getOneOrNullResult();

    }

    public function findOneHighlighted($locale): ?Category
    {
        return $this->createQueryBuilder('a')
            ->leftJoin('a.translations', 'trans')
            //->andWhere('trans.isHighlighted = true')
            ->andWhere('trans.locale = :locale')
            ->andWhere('a.startDate <= :today')
            ->andWhere('a.endDate >= :today')
            ->setParameter('locale', $locale)
            ->setParameter('today', new \DateTime())
            ->setMaxResults(1)
            ->orderBy('a.startDate', 'DESC')
            ->getQuery()
            ->getOneOrNullResult();
    }

    public function findByApply($locale): array
    {
        return $this->createQueryBuilder('a')
            ->leftJoin('a.translations', 'trans')
            ->andWhere('trans.isActive = true')
            ->andWhere('a.isApply = true')
            ->andWhere('trans.locale = :locale')
            ->andWhere('a.startDate <= :today')
            ->andWhere('a.endDate >= :today')
            ->setParameter('locale', $locale)
            ->setParameter('today', new \DateTime())
            ->orderBy('a.startDate', 'ASC')
            //->setMaxResults(1)

            ->getQuery()
            ->getResult();
    }

    public function findAllByApply($locale): array
    {
        return $this->createQueryBuilder('a')
            ->leftJoin('a.translations', 'trans')
            ->andWhere('trans.isActive = true')
            ->andWhere('a.isApply = true')
            ->andWhere('trans.locale = :locale')
            ->setParameter('locale', $locale)
            ->orderBy('a.startDate', 'ASC')
            //->setMaxResults(1)

            ->getQuery()
            ->getResult();
    }

    public function getChart(): ?array
    {
        return $this->createQueryBuilder('e')
            ->select("DATE_FORMAT(e.createdAt, '%Y-%m') as x, COUNT(e) as count")
            ->groupBy('x')
            ->addOrderBy('x', 'ASC')
            ->getQuery()->getResult();
    }
}
