<?php

declare(strict_types=1);

namespace App\Repository\Education\Lesson;

use App\Entity\Education\Lesson\Lesson;
use App\Repository\Shared\PaginationAwareRepositoryTrait;
use App\Service\Pagination\Context\PaginationContext;
use App\Traits\Repository\Editable;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\ORM\QueryBuilder;
use Doctrine\Persistence\ManagerRegistry;

final class LessonRepository extends ServiceEntityRepository
{
    use Editable, PaginationAwareRepositoryTrait;

    public function __construct(ManagerRegistry $managerRegistry)
    {
        parent::__construct($managerRegistry, Lesson::class);
    }

    public function findPublics($categories): QueryBuilder
    {
        $queryBuilder = $this->createQueryBuilder('a')
            ->select('a','at','ac','act')
            ->innerJoin('a.translations', 'at')
            ->innerJoin('a.category', 'ac')
            ->innerJoin('ac.translations', 'act')
            ->andWhere('at.isActive = true')
           ;

        if ($categories) {

            $queryBuilder

                ->andWhere('act.slug IN(:categories)')
                ->setParameter('categories', $categories);
        }
        return $queryBuilder;
    }

    public function findOneBySlug($locale, $slug): ?Lesson
    {
        return $this->createQueryBuilder('a')
            ->leftJoin('a.translations', 'trans')
            ->andWhere('trans.slug = :slug')
            ->andWhere('trans.locale = :locale')
            ->setMaxResults(1)
            ->setParameter('slug', $slug)
            ->setParameter('locale', $locale)
            ->getQuery()
            ->getOneOrNullResult();

    }

    public function findHighlighteds($locale): array
    {
        return $this->createQueryBuilder('a')
            ->leftJoin('a.translations', 'trans')
            ->andWhere('trans.isHighlighted = true')
            ->andWhere('trans.locale = :locale')
            ->setParameter('locale', $locale)
            ->setMaxResults(3)
            ->getQuery()
            ->getResult();
    }

    public function getChart(): ?array
    {
        return $this->createQueryBuilder('e')
            ->select("DATE_FORMAT(e.createdAt, '%Y-%m') as x, COUNT(e) as count")
            ->groupBy('x')
            ->addOrderBy('x', 'ASC')
            ->getQuery()->getResult();
    }


    public function search($keyword): array
    {
        if ($keyword) {
            $queryBuilder = $this->createQueryBuilder('a')
                ->select('a','at','ac','act')
                ->innerJoin('a.translations', 'at')
                ->innerJoin('a.category', 'ac')
                ->innerJoin('ac.translations', 'act')
                ->andWhere('at.title LIKE :keyword')
                ->setParameter('keyword', '%' . $keyword . '%')
                ->andWhere('at.isActive = true')
            ;
            return $queryBuilder->getQuery()->getResult();
        }
        return [];

    }

}
