<?php

declare(strict_types=1);

namespace App\Repository\User;

use App\Entity\User\User;
use App\Traits\Repository\Editable;
use DateTimeInterface;
use Doctrine\Bundle\DoctrineBundle\Repository\ServiceEntityRepository;
use Doctrine\Persistence\ManagerRegistry;

final class UserRepository extends ServiceEntityRepository
{
    use Editable;

    public function __construct(ManagerRegistry $managerRegistry)
    {
        parent::__construct($managerRegistry, User::class);
    }

    public function getEntityClass(): string
    {
        return User::class;
    }

    public function findById(string $id): ?User
    {
        return $this->createQueryBuilder('u')
            ->andWhere('u.id = :id')
            ->setParameter('id', $id)
            ->getQuery()
            ->getOneOrNullResult();
    }

    public function findByFacebookId(string $id): ?User
    {
        return $this->createQueryBuilder('u')
            ->andWhere('u.facebookId = :id')
            ->setParameter('id', $id)
            ->getQuery()
            ->getOneOrNullResult();
    }

    public function findByGoogleId(string $id): ?User
    {
        return $this->createQueryBuilder('u')
            ->andWhere('u.googleId = :googleId')
            ->setParameter('googleId', $id)
            ->getQuery()
            ->getOneOrNullResult();
    }

    public function findByEmail(string $email): ?User
    {
        return $this->createQueryBuilder('u')
            ->andWhere('u.email = :email')
            ->setParameter('email', $email)
            ->getQuery()
            ->getOneOrNullResult();
    }

    public function findByVerificationToken(string $token): ?User
    {
        return $this->createQueryBuilder('u')
            ->andWhere('u.verificationToken = :token')
            ->setParameter('token', $token)
            ->getQuery()
            ->getOneOrNullResult();
    }

    /**
     * @return array<User>
     */
    public function findRegisteredBetweenAndNotVerified(DateTimeInterface $from, DateTimeInterface $to): array
    {
        return $this->createQueryBuilder('u')
            ->andWhere('u.createdAt > :from')
            ->andWhere('u.createdAt < :to')
            ->andWhere('u.verified = :verified')
            ->setParameter('from', $from)
            ->setParameter('to', $to)
            ->setParameter('verified', false)
            ->getQuery()
            ->getResult();
    }

    public function getChart(): ?array
    {
        return $this->createQueryBuilder('e')
            ->select("DATE_FORMAT(e.createdAt, '%Y-%m') as x, COUNT(e) as count")
            ->groupBy('x')
            ->addOrderBy('x', 'ASC')
            ->getQuery()->getResult();
    }
}
