<?php

declare(strict_types=1);

namespace App\Security\Authenticator\Portal;

use App\Entity\User\User;
use App\Enum\User\RegistrationSource;
use App\Event\User\UserRegisteredEvent;
use App\Exception\Security\InvalidActionException;
use App\Factory\User\UserFactory;
use App\Repository\User\UserRepository;
use App\Service\Social\FacebookClient;
use App\Service\Social\FacebookUserUpdater;
use KnpU\OAuth2ClientBundle\Security\Authenticator\OAuth2Authenticator;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Security\Core\Authentication\Token\TokenInterface;
use Symfony\Component\Security\Core\Exception\AuthenticationException;
use Symfony\Component\Security\Http\Authentication\AuthenticationFailureHandlerInterface;
use Symfony\Component\Security\Http\Authentication\AuthenticationSuccessHandlerInterface;
use Symfony\Component\Security\Http\Authenticator\Passport\Badge\UserBadge;
use Symfony\Component\Security\Http\Authenticator\Passport\Passport;
use Symfony\Component\Security\Http\Authenticator\Passport\SelfValidatingPassport;
use Symfony\Component\Security\Http\EntryPoint\AuthenticationEntryPointInterface;

final class FacebookAuthenticator extends OAuth2Authenticator implements AuthenticationEntryPointInterface
{
    public function __construct(
        private readonly FacebookClient               $facebookClient,
        private readonly UserFactory                  $userFactory,
        private readonly FacebookUserUpdater          $facebookUserUpdater,
        private readonly UserRepository               $userRepository,
        private readonly EventDispatcherInterface     $eventDispatcher,
        private readonly AuthenticationSuccessHandlerInterface $authenticationSuccessHandler,
        private readonly AuthenticationFailureHandlerInterface $authenticationFailureHandler
    )
    {
    }

    public function supports(Request $request): ?bool
    {
        return $request->attributes->get('_route') === 'portal_api_auth_login_facebook_verify';
    }

    public function start(Request $request, AuthenticationException $authException = null): Response
    {
        throw new InvalidActionException();
    }

    public function authenticate(Request $request): Passport
    {
        $accessToken = $this->fetchAccessToken($this->facebookClient);
        return new SelfValidatingPassport(new UserBadge($accessToken->getToken(), function () use ($accessToken): User {
            $facebookUser = $this->facebookClient->fetchUserFromToken($accessToken);

            $user = $this->userRepository->findByGoogleId($facebookUser->getId());

            if (!$user instanceof User) {
                $user = $this->userRepository->findByEmail($facebookUser->getEmail());
            }

            $registration = false;

            if (!$user instanceof User) {
                $user = $this->userFactory->create($facebookUser->getEmail());
                $registration = true;
            }

            $this->facebookUserUpdater->update($user, $facebookUser);

            $this->userRepository->add($user, true);

            if ($registration) {
                $this->eventDispatcher->dispatch(new UserRegisteredEvent($user, RegistrationSource::Facebook));
            }

            return $user;
        }));
    }

    public function onAuthenticationSuccess(Request $request, TokenInterface $token, string $firewallName): ?Response
    {
        return $this->authenticationSuccessHandler->onAuthenticationSuccess($request, $token);
    }

    public function onAuthenticationFailure(Request $request, AuthenticationException $exception): ?Response
    {
        return $this->authenticationFailureHandler->onAuthenticationFailure($request, $exception);
    }
}
