<?php

declare(strict_types=1);

namespace App\Table\Abstract;

use AllowDynamicProperties;
use App\Entity\Interfaces\TranslatableInterface;
use Doctrine\ORM\QueryBuilder;
use Omines\DataTablesBundle\Adapter\Doctrine\ORMAdapter;
use Omines\DataTablesBundle\Column\TextColumn;
use Omines\DataTablesBundle\Column\TwigStringColumn;
use Omines\DataTablesBundle\DataTable;
use Omines\DataTablesBundle\DataTableTypeInterface;
use Twig\Environment;

#[AllowDynamicProperties]
abstract class BaseTable implements DataTableTypeInterface
{
    public function __construct(
        public readonly Environment $twig
    )
    {
    }

    public function configure(DataTable $dataTable, array $options): void
    {
        $entity = new $options['entity'];

        $dataTable->add('uuid', TextColumn::class, ['visible' => false, 'searchable' => false]);

        if (method_exists($entity, 'IsActive')) {
            $dataTable
                ->add('isActive', TextColumn::class, ['render' => fn($value, $context): string => $this->twig->render('system/components/table/_state.html.twig', [
                    'state' => $context->isActive()
                ]), 'searchable' => false])
                ->add('isHighlighted', TextColumn::class, ['render' => fn($value, $context): string => $this->twig->render('system/components/table/_state.html.twig', [
                    'state' => $context->isHighlighted()
                ]), 'searchable' => false])
              ;
        }
        $dataTable->add('createdAt', TwigStringColumn::class, ['template' => '{{ (row.updatedAt) ? row.updatedAt|date("Y m d. H:i") }}', 'searchable' => false]);

        if (method_exists($entity, 'getUpdatedBy') || $entity->getTranslationEntityClass()) {
            $dataTable
                ->add('updatedAt', TwigStringColumn::class, ['template' => '{{ (row.updatedAt) ? row.updatedAt|date("Y-m-d H:i") }}', 'searchable' => false]);
        }

        $dataTable
            ->add('actions', TextColumn::class, [
                'render' => fn($value, $context): string => $this->twig->render('system/components/table/_actions.html.twig', [
                    'meta' => ['route_prefix' => $options['prefix']],
                    'object' => $context
                ]),
                'searchable' => false,
                'orderable' => false,
                'label' => ''
            ])
            ->addOrderBy('createdAt', DataTable::SORT_DESCENDING);


        if ($entity instanceof TranslatableInterface) {
            $dataTableContext = [
                'entity' => $entity->getTranslationEntityClass(),
                'query' => function (QueryBuilder $queryBuilder) use ($entity, $options): void {
                    $queryBuilder
                        ->select('trans, a')
                        ->from($entity->getTranslationEntityClass(), 'trans')
                        ->join('trans.translatable', 'a')
                        ->where('trans.locale = :locale')
                        ->setParameter('locale', $options['locale']);
                }
            ];
        } else {
            $dataTableContext = [
                'entity' => $options['entity'],
                'query' => function (QueryBuilder $queryBuilder) use ($options): void {
                    $queryBuilder
                        ->select('a')
                        ->from($options['entity'], 'a');
                }
            ];
        }

        $dataTable->createAdapter(ORMAdapter::class, $dataTableContext);
    }

    public function preConfigure(DataTable $dataTable, array $options): DataTable
    {
        $entity = new $options['entity'];

        if (method_exists($entity, 'getTitle') || method_exists($entity, 'getTranslationEntityClass')) {
            $dataTable
                ->add('title', TextColumn::class, [
                    'field' => 'trans.title',
                    'render' => fn($value, $context): string => $this->twig->render('system/components/table/_title.html.twig', [
                        'meta' => ['route_prefix' => $options['prefix']],
                        'object' => $context
                    ])
                ]);
        }


        return $dataTable;
    }
}
