<?php

declare(strict_types=1);

namespace App\Traits\Entity;

use App\Entity\Interfaces\TranslationInterface;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\Common\Collections\Collection;

trait Translatable
{
    protected ?Collection $translations = null;

    protected ?Collection $newTranslations = null;

    protected string $currentLocale;

    protected string $defaultLocale = 'en';


    public function __call($method, $arguments): mixed
    {
        if (!method_exists(self::getTranslationEntityClass(), $method)) {
            $method = 'get' . ucfirst((string)$method);
        }

        $translation = $this->translate($this->getCurrentLocale(), false);

        return call_user_func_array([$translation, $method], $arguments);
    }

    public static function getTranslationEntityClass(): string
    {
        return static::class . 'Translation';
    }

    public function translate(?string $locale = null, bool $fallbackToDefault = true): TranslationInterface
    {
        return $this->doTranslate($locale, $fallbackToDefault);
    }

    protected function doTranslate(?string $locale = null, bool $fallbackToDefault = true): TranslationInterface
    {
        if ($locale === null) {
            $locale = $this->getCurrentLocale();
        }

        $foundTranslation = $this->findTranslationByLocale($locale);
        if ($foundTranslation && !$foundTranslation->isEmpty()) {
            return $foundTranslation;
        }

        if ($fallbackToDefault) {
            $fallbackTranslation = $this->resolveFallbackTranslation($locale);
            if ($fallbackTranslation !== null) {
                return $fallbackTranslation;
            }
        }

        if ($foundTranslation) {
            return $foundTranslation;
        }

        $translationEntityClass = static::getTranslationEntityClass();

        /** @var TranslationInterface $translation */
        $translation = new $translationEntityClass();
        $translation->setLocale($locale);

        $this->getNewTranslations()
            ->set($translation->getLocale(), $translation);
        $translation->setTranslatable($this);

        return $translation;
    }

    public function getCurrentLocale(): string
    {
        return $this->currentLocale ?: $this->getDefaultLocale();
    }

    public function setCurrentLocale(string $locale): void
    {
        $this->currentLocale = $locale;
    }

    public function getDefaultLocale(): string
    {
        return $this->defaultLocale;
    }

    public function setDefaultLocale(string $locale): void
    {
        $this->defaultLocale = $locale;
    }

    protected function findTranslationByLocale(string $locale, bool $withNewTranslations = true): ?TranslationInterface
    {
        $translation = $this->getTranslations()
            ->get($locale);

        if ($translation) {
            return $translation;
        }

        if ($withNewTranslations) {
            return $this->getNewTranslations()
                ->get($locale);
        }

        return null;
    }

    public function getTranslations(): Collection
    {
        if ($this->translations === null) {
            $this->translations = new ArrayCollection();
        }

        return $this->translations;
    }

    public function setTranslations(iterable $translations): void
    {
        $this->ensureIsIterableOrCollection($translations);

        foreach ($translations as $translation) {
            $this->addTranslation($translation);
        }
    }

    public function getNewTranslations(): Collection
    {
        if ($this->newTranslations === null) {
            $this->newTranslations = new ArrayCollection();
        }

        return $this->newTranslations;
    }

    private function resolveFallbackTranslation(string $locale): ?TranslationInterface
    {
        $fallbackLocale = $this->computeFallbackLocale($locale);

        if ($fallbackLocale !== null) {
            $translation = $this->findTranslationByLocale($fallbackLocale);
            if ($translation && !$translation->isEmpty()) {
                return $translation;
            }
        }

        return $this->findTranslationByLocale($this->getDefaultLocale(), false);
    }

    protected function computeFallbackLocale(string $locale): ?string
    {
        if (strrchr($locale, '_') !== false) {
            return substr($locale, 0, -strlen(strrchr($locale, '_')));
        }

        return null;
    }

    private function ensureIsIterableOrCollection($translations): void
    {
        if ($translations instanceof Collection) {
            return;
        }

        if (is_iterable($translations)) {
            return;
        }

        throw new \Exception(
            sprintf('$translations parameter must be iterable or %s', Collection::class)
        );
    }

    public function addTranslation(TranslationInterface $translation): void
    {
        $this->getTranslations()->set($translation->getLocale(), $translation);
        $translation->setTranslatable($this);
    }

    public function mergeNewTranslations(): void
    {
        foreach ($this->getNewTranslations() as $newTranslation) {
            if (!$this->getTranslations()->contains($newTranslation) && !$newTranslation->isEmpty()) {
                $this->addTranslation($newTranslation);
                $this->getNewTranslations()->removeElement($newTranslation);
            }
        }

        foreach ($this->getTranslations() as $translation) {
            if (!$translation->isEmpty()) {
                continue;
            }

            $this->removeTranslation($translation);
        }
    }

    public function removeTranslation(TranslationInterface $translation): void
    {
        $this->getTranslations()->removeElement($translation);
    }

    protected function proxyCurrentLocaleTranslation(string $method, array $arguments = []): mixed
    {
        if (!method_exists(self::getTranslationEntityClass(), $method)) {
            $method = 'get' . ucfirst($method);
        }

        $translation = $this->translate($this->getCurrentLocale());

        return call_user_func_array([$translation, $method], $arguments);
    }
}
