<?php

declare(strict_types=1);

/*
 * This file is part of the TranslationFormBundle package.
 *
 * (c) David ALLIX <http://a2lix.fr>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

namespace A2lix\TranslationFormBundle\Tests\Form\Type;

use A2lix\TranslationFormBundle\Form\Type\TranslationsFormsType;
use A2lix\TranslationFormBundle\Tests\Fixtures\Entity\MediaLocalize;
use A2lix\TranslationFormBundle\Tests\Fixtures\Entity\Product;
use A2lix\TranslationFormBundle\Tests\Fixtures\Form\MediaLocalizeType;
use A2lix\TranslationFormBundle\Tests\Form\TypeTestCase;
use Symfony\Component\Form\Extension\Core\Type\FormType;
use Symfony\Component\Form\Extension\Core\Type\SubmitType;
use Symfony\Component\Form\PreloadedExtension;

/**
 * @internal
 */
final class TranslationsFormsTypeSimpleTest extends TypeTestCase
{
    protected $locales = ['en', 'fr', 'de'];
    protected $defaultLocale = 'en';
    protected $requiredLocales = ['en', 'fr'];

    public function testEmptyForm(): void
    {
        $form = $this->factory->createBuilder(FormType::class, new Product())
            ->add('url')
            ->add('medias', TranslationsFormsType::class, [
                'form_type' => MediaLocalizeType::class,
            ])
            ->add('save', SubmitType::class)
            ->getForm()
        ;

        $mediasForm = $form->get('medias')->all();
        $mediasLocales = array_keys($mediasForm);
        $mediasRequiredLocales = array_keys(array_filter($mediasForm, function ($form) {
            return $form->isRequired();
        }));

        static::assertEquals($this->locales, $mediasLocales, 'Locales should be same as config');
        static::assertEquals($this->requiredLocales, $mediasRequiredLocales, 'Required locales should be same as config');

        static::assertEquals(['url', 'description'], array_keys($mediasForm['en']->all()), 'Fields should matches MediaLocalizeType fields');
        static::assertEquals(['url', 'description'], array_keys($mediasForm['fr']->all()), 'Fields should matches MediaLocalizeType fields');
        static::assertEquals(['url', 'description'], array_keys($mediasForm['de']->all()), 'Fields should matches MediaLocalizeType fields');
    }

    public function testCreationForm(): Product
    {
        $form = $this->factory->createBuilder(FormType::class, new Product())
            ->add('url')
            ->add('medias', TranslationsFormsType::class, [
                'form_type' => MediaLocalizeType::class,
            ])
            ->add('save', SubmitType::class)
            ->getForm()
        ;

        $mediaEn = new MediaLocalize();
        $mediaEn->setLocale('en')
            ->setUrl('http://en')
            ->setDescription('desc en')
        ;
        $mediaFr = new MediaLocalize();
        $mediaFr->setLocale('fr')
            ->setUrl('http://fr')
            ->setDescription('desc fr')
        ;
        $mediaDe = new MediaLocalize();
        $mediaDe->setLocale('de')
            ->setUrl('http://de')
            ->setDescription('desc de')
        ;

        $product = new Product();
        $product->setUrl('a2lix.fr')
            ->addMedia($mediaEn)
            ->addMedia($mediaFr)
            ->addMedia($mediaDe)
        ;

        $formData = [
            'url' => 'a2lix.fr',
            'medias' => [
                'en' => [
                    'url' => 'http://en',
                    'description' => 'desc en',
                ],
                'fr' => [
                    'url' => 'http://fr',
                    'description' => 'desc fr',
                ],
                'de' => [
                    'url' => 'http://de',
                    'description' => 'desc de',
                ],
            ],
        ];

        $form->submit($formData);
        static::assertTrue($form->isSynchronized());
        static::assertEquals($product, $form->getData());

        return $product;
    }

    /**
     * @depends testCreationForm
     */
    public function testEditionForm(Product $product): void
    {
        $product->getMedias()['en']->setUrl('http://ennnnn');
        $product->getMedias()['fr']->setDescription('desc frrrrrr');

        $formData = [
            'url' => 'a2lix.fr',
            'medias' => [
                'en' => [
                    'url' => 'http://ennnnn',
                    'description' => 'desc en',
                ],
                'fr' => [
                    'url' => 'http://fr',
                    'description' => 'desc frrrrrr',
                ],
                'de' => [
                    'url' => 'http://de',
                    'description' => 'desc de',
                ],
            ],
        ];

        $form = $this->factory->createBuilder(FormType::class, new Product())
            ->add('url')
            ->add('medias', TranslationsFormsType::class, [
                'form_type' => MediaLocalizeType::class,
            ])
            ->add('save', SubmitType::class)
            ->getForm()
        ;

        $form->submit($formData);
        static::assertTrue($form->isSynchronized());
        static::assertEquals($product, $form->getData());

        $view = $form->createView();
        $children = $view->children;

        foreach (array_keys($formData) as $key) {
            static::assertArrayHasKey($key, $children);
        }
    }

    protected function getExtensions(): array
    {
        $translationsFormsType = $this->getConfiguredTranslationsFormsType($this->locales, $this->defaultLocale, $this->requiredLocales);

        return [new PreloadedExtension([
            $translationsFormsType,
        ], [])];
    }
}
