<?php

/*
 * This file is part of the Symfony package.
 *
 * (c) Fabien Potencier <fabien@symfony.com>
 *
 * For the full copyright and license information, please view the LICENSE
 * file that was distributed with this source code.
 */

use Symfony\Component\Config\Loader\LoaderInterface;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Kernel;

/**
 * App Test Kernel for functional tests.
 *
 * @author Johannes M. Schmitt <schmittjoh@gmail.com>
 */
class AppKernel extends Kernel
{
    private $testCase;
    private $rootConfig;

    public function __construct($testCase, $debug = true)
    {
        $environment = $testCase;

        if (!is_dir(__DIR__ . '/' . $testCase)) {
            throw new \InvalidArgumentException(sprintf('The test case "%s" does not exist.', $testCase));
        }
        $this->testCase = $testCase;

        $rootConfig = __DIR__ . '/' . $testCase . '/config.yml';

        $fs = new Filesystem();
        if (!$fs->isAbsolutePath($rootConfig)
            && !file_exists(
                $rootConfig = __DIR__ . '/' . $testCase . '/' . $rootConfig
            )
        ) {
            throw new \InvalidArgumentException(sprintf('The root config "%s" does not exist.', $rootConfig));
        }
        $this->rootConfig = $rootConfig;

        parent::__construct($environment, $debug);
    }

    public function registerBundles(): iterable
    {
        if (!file_exists($filename = $this->getRootDir() . '/' . $this->testCase . '/bundles.php')) {
            throw new \RuntimeException(sprintf('The bundles file "%s" does not exist.', $filename));
        }

        return include $filename;
    }

    public function getRootDir()
    {
        return __DIR__;
    }

    public function getCacheDir(): string
    {
        return $this->getRootDir() . '/tmp/' . $this->testCase . '/cache/' . $this->environment;
    }

    public function getLogDir(): string
    {
        return $this->getRootDir() . '/tmp/' . $this->testCase . '/logs';
    }

    public function registerContainerConfiguration(LoaderInterface $loader)
    {
        $loader->load($this->rootConfig);
    }

    public function serialize()
    {
        return serialize(array($this->testCase, $this->rootConfig, $this->getEnvironment(), $this->isDebug()));
    }

    public function unserialize($str)
    {
        call_user_func_array(array($this, '__construct'), unserialize($str));
    }

    protected function getKernelParameters(): array
    {
        $parameters = parent::getKernelParameters();
        $parameters['kernel.test_case'] = $this->testCase;

        return $parameters;
    }

    public function terminate(
        Request $request, Response $response)
    {
        parent::terminate($request, $response); // TODO: Change the autogenerated stub
    }
}
